# convertBasicLabelToLineNr
# V1.0 2025-10-20
#
# add line numbers to a besic program with labels used for
# argument for one of the following basic commands:
# GOTO, GOSUB, THEN, RESTORE, ARUN, AUTORUN
#
# REM this is a test
# :_L100
# PRINT "Hello ";
# GOTO _L100
#
# will be converted to
#
# 100 REM this is a test
# 120 PRINT "Hello ";
# 120 GOTO 110
#
# a label is defined within its own line, starting with a colon followed
# by at least one char, number or underscore
# labels meight no use the name of a basic command
# 
# usage:
# cat "C:\sharp\test.sbq" | powershell -File ".\convertBasicLabelToLineNr.ps1"
# this uses
# Infile = STDIN
# Outfile = STDOUT
#
# or
# 
# powershell "C:\sharp\convertBasicLabelToLineNr.ps1" -Infile "C:\sharp\test.sbq" -Outfile "-"
# powershell "C:\sharp\convertBasicLabelToLineNr.ps1" -Infile "C:\sharp\test.sbq" -Outfile "C:\sharp\test_with_linenr.sbq"
# Infile = <path and filename to the source file>
# Outfile = STDOUT or <path and filename to the destination file>
#
# note the usage of "-File" in the upper example when using STDIN
#
# Copyright 2025 Bernhard Schwall
# The file is released with no waranty under the BSD-3-Clause License (see end of file for the whole license)

Param (
  [string]$Infile = '-',
  [string]$Outfile = '-'
)

$FirstLineNr = 100
$StepLineNr = 10
$AddColonToLineNr = $true

# Aufruf entweder mit
# cat "C:\sharp\test.sbq" | powershell -File "C:\sharp\renumber-sharp.ps1"
# dann gilt:
# Infile = STDIN
# Outfile = STDOUT

# ODER
# powershell "C:\sharp\renumber-sharp.ps1" -Infile "C:\sharp\test.sbq" -Outfile "-"
# powershell "C:\sharp\renumber-sharp.ps1" -Infile "C:\sharp\test.sbq" -Outfile "C:\sharp\test_renumbered.sbq"
# Infile = Datei
# Outfile = STDOUT oder Datei
#
# man beachte das -File oder nicht -File, je nachdem, was man mchte

$code = @"
  Imports System
  Imports System.Text
  Imports System.Collections
  Imports System.Collections.Generic

  Namespace Sharp
    Public Class SharpBasic

  Public Function convertBasicLabelToLineNr(ByVal sText As String(), ByVal iFirstLineNr As Integer, ByVal iStepLineNr As Integer, ByVal bAddColonToLineNr As Boolean) As String
    Dim sbResult As StringBuilder
    Dim liZeilen As List(Of tZeile)
    Dim Zeile As tZeile
    Dim htLabel As Hashtable ' list of labels and their (new) line numbers
    Dim iLine As Integer
    Dim iLineNr As Integer
    Dim iLineStep As Integer
    Dim sLine As String
    Dim iPos As Integer
    Dim liCommands As List(Of String)
    Dim iCommand As Integer

    liCommands = New List(Of String)
    liCommands.Add("GOTO")
    liCommands.Add("GOSUB")
    liCommands.Add("RESTORE")
    liCommands.Add("THEN")
    liCommands.Add("ARUN")
    liCommands.Add("AUTOGOTO")

    liZeilen = New List(Of tZeile)
    htLabel = New Hashtable

    ' Pass 1: identify all label and generate new line numbers
    iLine = 0
    iLineNr = iFirstLineNr
    iLineStep = iStepLineNr
    While (iLine < sText.Length)
      sLine = sText(iLine).Trim
      If sLine.Length > 0 Then
        Zeile = New tZeile
        If sLine.StartsWith(":") Then
          If sLine.Length < 2 Then
            Return "Fehler in Zeile " & (iLine + 1) & " : Keine Label nach : gefunden"
          End If
          sLine = sLine.Substring(1)
          If htLabel.ContainsKey(sLine) Then
            Return "Fehler in Zeile " & (iLine + 1) & " : Label <" & sLine & "> ist mehrfach vergeben"
          Else
            htLabel.Add(sLine, iLineNr)
          End If
        Else
          Zeile.iLineNr = iLineNr
          Zeile.sContent = sLine
          liZeilen.Add(Zeile)
          iLineNr += iLineStep
        End If
      End If
      iLine += 1
    End While

    ' htLabel contains all labels and their new line numbers
    ' liZeilen contains all lines (excapt the ones with Labels) from the program with new line number arguments

    ' Pass 2: identify commands from liCommands enthalten and replance label arguments with new line numbers
    iLine = 0
    sbResult = New StringBuilder
    While iLine < liZeilen.Count
      sLine = liZeilen(iLine).sContent
      iPos = 0
      If bAddColonToLineNr Then
        sbResult.Append(String.Format("{0:0}: ", liZeilen(iLine).iLineNr))
      Else
        sbResult.Append(String.Format("{0:0} ", liZeilen(iLine).iLineNr))
      End If
      While iPos < sLine.Length
        iCommand = FindCommand(sLine, iPos, liCommands)
        If iCommand > -1 Then
          ReplaceLabelToLineNr(sLine, iPos, htLabel)
        End If
      End While
      sbResult.AppendLine(sLine)
      iLine += 1
    End While

    Return sbResult.ToString
  End Function

  Private Class tZeile
    Public iLineNr As Integer
    Public sLabel As String
    Public sContent As String
    Public Sub New()
      iLineNr = 0
      sLabel = ""
      sContent = ""
    End Sub
  End Class

  ' htLabel contains:
  ' Key = label (should not be a basic command)
  ' Value = new line number
  Private Sub ReplaceLabelToLineNr(ByRef sLine As String, ByRef iPos As Integer, ByVal htLabel As Hashtable)
    Dim iPosOld As Integer
    Dim sCommand As String
    Dim sLabel As String
    Dim sLineNew As String
    Dim sLineNr As String
    While iPos < sLine.Length
      iPosOld = iPos
      sCommand = ReadNextCommand(sLine, iPos)
      If sCommand.Length = 0 Then
        Exit While
      ElseIf sCommand.StartsWith("""") Then
        ' quoted label
      ElseIf htLabel.ContainsKey(sCommand) Then
        ' label found, replace it by the line number
        sLabel = sCommand
        sLineNr = htLabel(sLabel)
        While iPosOld < sLine.Length AndAlso sLine(iPosOld) = " "
          iPosOld += 1
        End While
        sLineNew = sLine.Substring(0, iPosOld) & sLineNr & sLine.Substring(iPos)
        sLine = sLineNew
        ' correct iPos
        iPos = iPosOld + sLineNr.Length
      Else
        ' keinen Label gefunden
        iPos = iPosOld
        Exit While
      End If
      ' prfen, ob SPC und "," folgt
      If Not IsCommaFollowing(sLine, iPos) Then
        Exit While
      End If
    End While
  End Sub

  ' test if next char is a comma (SPC are skipped)
  Private Function IsCommaFollowing(sLine As String, ByRef iPos As Integer) As Boolean
    While iPos < sLine.Length AndAlso sLine(iPos) = " "
      iPos += 1
    End While
    If iPos < sLine.Length AndAlso sLine(iPos) = "," Then
      iPos += 1
      Return True
    End If
    Return False
  End Function

  ' search sLine for a command in liCommands, skip text in quotation marks and ignore chars not beeing a..z, numbers or quoted text
  ' Out:
  '   return: >= 0 id of an identified command in liCommands
  '                iPos = char after this command
  '           -1   no command found
  '                iPos = end of line (sLine.Length)
  Private Function FindCommand(sLine As String, ByRef iPos As Integer, liCommands As List(Of String)) As Integer
    Dim iCommand As Integer = -1
    Dim sCommand As String

    sLine = sLine.ToUpper ' we only search for strings, so case is ignored

    While iPos < sLine.Length
      sCommand = ReadNextCommand(sLine, iPos)
      If sCommand.Length = 0 Then
        ' skip each char that is no text, number oder quoted text
        iPos += 1
      ElseIf sCommand.StartsWith("""") = False Then
        ' iPos points to the char after sCommand
        For iCommand = 0 To liCommands.Count - 1
          If liCommands(iCommand) = sCommand Then
            ' an command is found
            Exit For
          End If
        Next
        If iCommand < liCommands.Count Then
          Exit While
        End If
      End If
    End While
    If iCommand = liCommands.Count Then
      ' nothing found
      iCommand = -1
    End If
    Return iCommand
  End Function

  ' read the next string/number/"label"
  Private Function ReadNextCommand(sLine As String, ByRef iPos As Integer) As String
    Dim sCommand As String = ""
    Dim iPosStart As Integer

    While iPos < sLine.Length AndAlso sLine(iPos) = " "
      iPos += 1
    End While

    iPosStart = iPos

    If iPos < sLine.Length AndAlso sLine(iPos) = """" Then
      ' skip the whole quoted string
      iPos += 1
      While iPos < sLine.Length AndAlso sLine(iPos) <> """"
        iPos += 1
      End While
      If iPos < sLine.Length Then
        ' closing quotation mark
        iPos += 1
      End If
    Else
      While iPos < sLine.Length AndAlso
      ((sLine(iPos) >= "0" And sLine(iPos) <= "9") Or
       (sLine(iPos) >= "A" And sLine(iPos) <= "Z") Or
       (sLine(iPos) >= "a" And sLine(iPos) <= "z") Or
       (sLine(iPos) = "_"))
        iPos += 1
      End While
    End If

    If iPos = iPosStart Then
      sCommand = ""
    ElseIf iPos < sLine.Length Then
      sCommand = sLine.Substring(iPosStart, iPos - iPosStart)
    Else
      sCommand = sLine.Substring(iPosStart)
    End If
    Return sCommand
  End Function

    End Class

  End Namespace
"@

if ($Infile -eq '-') {
  # STDIN nach $inputText lesen
  $inputText = [System.Collections.ArrayList]@()
  foreach ($o in $input) {
    $inputText.Add($o) | Out-Null
  }
} else {
  if (Test-Path -Path $Infile -PathType Leaf) {
    $inputText = Get-Content -Path $Infile -Encoding UTF8
  } else {
    'Input-Datei <{0}> nicht gefunden.' -f $Infile
  }
}

if ($Outfile -ne '-') {
  # Dateinamen angegeben
  if (Test-Path -Path $Outfile -PathType Leaf) {
    Remove-Item -Path $Outfile
  }
}

if (-not ([System.Management.Automation.PSTypeName]'Sharp.SharpBasic').Type)
{
    Add-Type -TypeDefinition $code -Language VisualBasic;
}

$myObj = new-object Sharp.SharpBasic
$res = $myObj.convertBasicLabelToLineNr($inputText, $FirstLineNr, $StepLineNr, $AddColonToLineNr)
if ($Outfile -eq '-') {
  $res
} else {
  $res | Out-File -FilePath $Outfile -Encoding utf8
}

#########################################################################
# License Notice:
# BSD-3-Clause License
#
# Copyright 2025 Bernhard Schwall
# 
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
# 
# 1. Redistributions of source code must retain the above copyright notice,
# this list of conditions and the following disclaimer.
# 
# 2. Redistributions in binary form must reproduce the above copyright notice,
# this list of conditions and the following disclaimer in the documentation
# and/or other materials provided with the distribution.
# 
# 3. Neither the name of the copyright holder nor the names of its contributors
# may be used to endorse or promote products derived from this software without
# specific prior written permission.
# 
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS AS IS
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.
# IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY
# DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
# (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
# LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
# ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
# THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
