# convertBasicLineNrToLabel
# V1.0 2025-10-20
#
# renumber a basic program to a program without line nmbers but with labels for each line number that is an
# argument for one of the following basic commands:
# GOTO, GOSUB, THEN, RESTORE, ARUN, AUTORUN
#
# 90 REM this is a test
# 100 PRINT "Hello ";
# 110 GOTO 100
#
# will be converted to
#
# REM this is a test
# :_L100
# PRINT "Hello ";
# GOTO _L100
# 
# usage:
# cat "C:\sharp\test.sbq" | powershell -File ".\convertBasicLineNrToLabel.ps1"
# this uses
# Infile = STDIN
# Outfile = STDOUT
#
# or
# 
# powershell "C:\sharp\convertBasicLineNrToLabel.ps1" -Infile "C:\sharp\test.sbq" -Outfile "-"
# powershell "C:\sharp\convertBasicLineNrToLabel.ps1" -Infile "C:\sharp\test.sbq" -Outfile "C:\sharp\test_renumbered.sbq"
# Infile = <path and filename to the original basic program>
# Outfile = STDOUT or <path and filename to the resulting basic program>
#
# note the usage of "-File" in the upper example when using STDIN
#
# Copyright 2025 Bernhard Schwall
# The file is released with no waranty under the BSD-3-Clause License (see end of file for the whole license)

Param (
  [string]$Infile = '-',
  [string]$Outfile = '-'
)

$code = @"
  Imports System
  Imports System.Text
  Imports System.Collections
  Imports System.Collections.Generic

  Namespace Sharp
    Public Class SharpBasic

  Public Function convertBasicLineNrToLabel(sText As String()) As String
    Dim sbResult As StringBuilder
    Dim liZeilen As List(Of tZeile)
    Dim Zeile As tZeile
    Dim htLineNumber As Hashtable ' list of line numbers used for jumps/restore
    Dim iLine As Integer
    Dim iLineNr As Integer
    Dim sLine As String
    Dim iPos As Integer
    Dim iPosContent As Integer
    Dim liCommands As List(Of String)
    Dim iCommand As Integer
    Dim sCommand As String

    liCommands = New List(Of String)
    liCommands.Add("GOTO")
    liCommands.Add("GOSUB")
    liCommands.Add("RESTORE")
    liCommands.Add("THEN")
    liCommands.Add("ARUN")
    liCommands.Add("AUTOGOTO")

    liZeilen = New List(Of tZeile)
    htLineNumber = New Hashtable

    ' Pass 1: identify all lines containing a command from liCommands and replance line number arguments with labels.
    iLine = 0
    While (iLine < sText.Length)
      sLine = sText(iLine).Trim
      If sLine.Length > 0 Then
        iPos = 0
        Zeile = New tZeile
        sCommand = ReadNextCommand(sLine, iPos)
        If Integer.TryParse(sCommand, iLineNr) Then
          If iLineNr < 0 Then
            Return "Fehler in Zeile " & (iLine + 1) & " : Keine Zeilennummer gefunden"
          End If
          Zeile.iLineNr = iLineNr
        Else
          Return "Fehler in Zeile " & (iLine + 1) & " : Keine Zeilennummer gefunden"
        End If
        If iPos < sLine.Length AndAlso sLine(iPos) = ":" Then
          ' we accept line number with or without colon
          iPos += 1
        End If
        iPosContent = iPos
        If iPos < sLine.Length Then
          Zeile.sLabel = sLine.Substring(0, iPos).Trim
          While iPos < sLine.Length
            iCommand = FindCommand(sLine, iPos, liCommands)
            If iCommand > -1 Then
              ReplaceLineNrToLabel(sLine, iPos, htLineNumber)
            End If
          End While
          Zeile.sContent = sLine.Substring(iPosContent).Trim
        Else
          Zeile.sLabel = sLine
          Zeile.sContent = ""
        End If
        liZeilen.Add(Zeile)
      End If
      iLine += 1
    End While

    ' htLineNumber contains all (as arguments) used line numbers and the new labels
    ' liZeilen contains all lines from the program with trplaced line number arguments

    ' Pass 2
    ' add lables for lines that a targes from liCommands as additional lines
    sbResult = New StringBuilder
    For iLine = 0 To liZeilen.Count - 1
      iPos = 0
      If liZeilen(iLine).iLineNr < 0 Then
        ' kann hier eigentlich nicht auftreten, da in Pass1 bereits abgefangen
        Return "Fehler in Zeile " & (iLine + 1) & " : Keine Zeilennummer gefunden"
      End If
      If htLineNumber.ContainsKey(liZeilen(iLine).iLineNr) Then
        sbResult.AppendLine(":" & htLineNumber.Item(liZeilen(iLine).iLineNr))
      End If
      sbResult.AppendLine(liZeilen(iLine).sContent)
    Next

    Return sbResult.ToString
  End Function

  Private Class tZeile
    Public iLineNr As Integer
    Public sLabel As String
    Public sContent As String
    Public Sub New()
      iLineNr = 0
      sLabel = ""
      sContent = ""
    End Sub
  End Class

  ' replace line number arguments after a liCommands and add a new label to htLineNumber
  ' line number can be a single number or a comma separated list.
  ' Input:
  ' sLine: the whole line
  ' iPos : the position after the found liCommands
  ' htLineNumber: Hashtable with all currently found line numbers (arguments only) and generated labels
  ' Output:
  ' sLine: modified line with line number arguments replanced by a label _L<line number>
  ' iPos : position after the last found line number (can bei sLine.Length)
  ' htLineNumber: Hashtable with new found line numbers an generated label
  Private Sub ReplaceLineNrToLabel(ByRef sLine As String, ByRef iPos As Integer, ByRef htLineNumber As Hashtable)
    Dim iPosOld As Integer
    Dim sCommand As String
    Dim sLabel As String
    Dim sLineNew As String
    Dim iLineNr As Integer
    While iPos < sLine.Length
      iPosOld = iPos
      sCommand = ReadNextCommand(sLine, iPos)
      If sCommand.Length = 0 Then
        ' end of argument list found
        Exit While
      ElseIf sCommand.StartsWith("""") Then
        ' quoted label can be skipped
      ElseIf Integer.TryParse(sCommand, iLineNr) Then
        ' line number found
        If Not htLineNumber.ContainsKey(iLineNr) Then
          sLabel = String.Format("_L{0}", iLineNr)
          htLineNumber.Add(iLineNr, sLabel) ' name of the new label that is not used as variable name
        Else
          sLabel = htLineNumber.Item(iLineNr)
        End If
        ' replace line number by label
        If sLine(iPosOld) = " " Then
          iPosOld += 1
        End If
        sLineNew = sLine.Substring(0, iPosOld) & sLabel & sLine.Substring(iPos)
        sLine = sLineNew
        ' fix iPos
        iPos = iPosOld + sLabel.Length
      Else
        ' no numner or label (e.g. IF THEN GOSUB ...)
        iPos = iPosOld
        Exit While
      End If
      ' test if "," follows
      If Not IsCommaFollowing(sLine, iPos) Then
        Exit While
      End If
    End While
  End Sub

  ' test if next char is a comma (SPC are skipped)
  Private Function IsCommaFollowing(sLine As String, ByRef iPos As Integer) As Boolean
    While iPos < sLine.Length AndAlso sLine(iPos) = " "
      iPos += 1
    End While
    If iPos < sLine.Length AndAlso sLine(iPos) = "," Then
      iPos += 1
      Return True
    End If
    Return False
  End Function

  ' search sLine for a command in liCommands, skip text in quotation marks and ignore chars not beeing a..z, numbers or quoted text
  ' Out:
  '   return: >= 0 id of an identified command in liCommands
  '                iPos = char after this command
  '           -1   no command found
  '                iPos = end of line (sLine.Length)
  Private Function FindCommand(sLine As String, ByRef iPos As Integer, liCommands As List(Of String)) As Integer
    Dim iCommand As Integer = -1
    Dim sCommand As String

    sLine = sLine.ToUpper ' we only search for strings, so case is ignored

    While iPos < sLine.Length
      sCommand = ReadNextCommand(sLine, iPos)
      If sCommand.Length = 0 Then
        ' skip each char that is no text, number oder quoted text
        iPos += 1
      ElseIf sCommand.StartsWith("""") = False Then
        ' iPos points to the char after sCommand
        For iCommand = 0 To liCommands.Count - 1
          If liCommands(iCommand) = sCommand Then
            ' an command is found
            Exit For
          End If
        Next
        If iCommand < liCommands.Count Then
          Exit While
        End If
      End If
    End While
    If iCommand = liCommands.Count Then
      ' nothing found
      iCommand = -1
    End If
    Return iCommand
  End Function

  ' read the next string/number/"label"
  Private Function ReadNextCommand(sLine As String, ByRef iPos As Integer) As String
    Dim sCommand As String = ""
    Dim iPosStart As Integer

    While iPos < sLine.Length AndAlso sLine(iPos) = " "
      iPos += 1
    End While

    iPosStart = iPos

    If iPos < sLine.Length AndAlso sLine(iPos) = """" Then
      ' skip the whole quoted string
      iPos += 1
      While iPos < sLine.Length AndAlso sLine(iPos) <> """"
        iPos += 1
      End While
      If iPos < sLine.Length Then
        ' closing quotation mark
        iPos += 1
      End If
    Else
      While iPos < sLine.Length AndAlso
      ((sLine(iPos) >= "0" And sLine(iPos) <= "9") Or
       (sLine(iPos) >= "A" And sLine(iPos) <= "Z") Or
       (sLine(iPos) >= "a" And sLine(iPos) <= "z") Or
       (sLine(iPos) = "_"))
        iPos += 1
      End While
    End If

    If iPos = iPosStart Then
      sCommand = ""
    ElseIf iPos < sLine.Length Then
      sCommand = sLine.Substring(iPosStart, iPos - iPosStart)
    Else
      sCommand = sLine.Substring(iPosStart)
    End If
    Return sCommand
  End Function

    End Class

  End Namespace
"@

if ($Infile -eq '-') {
  # STDIN nach $inputText lesen
  $inputText = [System.Collections.ArrayList]@()
  foreach ($o in $input) {
    $inputText.Add($o) | Out-Null
  }
} else {
  if (Test-Path -Path $Infile -PathType Leaf) {
    $inputText = Get-Content -Path $Infile -Encoding UTF8
  } else {
    'Input-Datei <{0}> nicht gefunden.' -f $Infile
  }
}

if ($Outfile -ne '-') {
  # Dateinamen angegeben
  if (Test-Path -Path $Outfile -PathType Leaf) {
    Remove-Item -Path $Outfile
  }
}

if (-not ([System.Management.Automation.PSTypeName]'Sharp.SharpBasic').Type)
{
    Add-Type -TypeDefinition $code -Language VisualBasic;
}

$myObj = new-object Sharp.SharpBasic
$res = $myObj.convertBasicLineNrToLabel($inputText)
if ($Outfile -eq '-') {
  $res
} else {
  $res | Out-File -FilePath $Outfile -Encoding utf8
}

#########################################################################
# License Notice:
# BSD-3-Clause License
#
# Copyright 2025 Bernhard Schwall
# 
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
# 
# 1. Redistributions of source code must retain the above copyright notice,
# this list of conditions and the following disclaimer.
# 
# 2. Redistributions in binary form must reproduce the above copyright notice,
# this list of conditions and the following disclaimer in the documentation
# and/or other materials provided with the distribution.
# 
# 3. Neither the name of the copyright holder nor the names of its contributors
# may be used to endorse or promote products derived from this software without
# specific prior written permission.
# 
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS AS IS
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.
# IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY
# DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
# (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
# LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
# ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
# THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
